<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-project-innovation-initiative
* @subpackage models
*/

/** */
require_once 'imap_custom_flag_model.php';

/**
* Model to handle custom flags on emails in an individual or group inbox.  
* 
* IMAP allows custom flags to be added to email messages, but due to a bug in php, the information for the custom flag
* cannot be retrieved aferward.  Because of this, our flags will be stored in both IMAP and the database.  Messages will
* include a flag id in the header that maps to a database entry that includes more information about the flag. Note that
* it will be important to closely couple this functionality to avoid orphaned ids in the header or orphaned records in 
* the database.
*
* Note that message ids are not persistant.  They change each time that the message moves, including when a user moves
* a message from one folder to another.  (This is why we're storing the flag id in the message header instead of 
* storing the message id in the flag table.)
*
* Users should only be able to view, edit, or delete flags that they created.  The exception to this is group mailboxes;
* when a message is flagged in a group mailbox, any member of the group may view, edit, or delete it.
*
* @author           <bell_adam@bah.com>
* @author M. Gibbs <gibbs_margaret@bah.com>
*
* @package direct-project-innovation-initiative
* @subpackage models
*/
class Flag_model extends IMAP_custom_flag_model {
	protected $table_name = 'flags';
	protected $primary_key = 'id';
	protected $colors = array('red','blue','green','orange','pink','purple'); //MUST contain at least one color;
	protected $mailbox_group;
	
	const CUSTOM_FLAG_PREFIX = 'flag';
	
	/**
	* Establishes the IMAP connection, triggers error if IMAP connection is not available.
	*/
	function __construct() {
        parent::__construct();
		
		$this->mailbox_group = $this->session->userdata('mailbox_group');
		if(empty($this->mailbox_group)) { show_404(); } //we can only deal with flags within the inbox, so mailbox_group should always be set
    }

#TODO - REMOVE THIS WHEN WE'RE SURE NO ONE'S USING IT
	function set_flag_for_message($message_id, $values){
		$this->error->notice(get_class($this).'::set_flag_for_message() will soon be deprecated; please use '.get_class($this).'::set_for_message() instead');
		return parent::set_for_message($message_id, $values);
	}
	
	///////////////////
	// SEARCH
	///////////////////
	
	/* Extends parent to enforce that users can only view flags that they've created */
	function find($id_or_conditions = array(), $key_by = null){
		$this->db->where('mailbox_group', $this->mailbox_group);
		return parent::find($id_or_conditions, $key_by);
	}
	
	
	///////////////////
	// DATA MANGEMENT
	///////////////////
	
	protected function _run_before_create(&$values){
		if(!parent::_run_before_create($values)) return false;
		if(empty($values['mailbox_group']))
			$values['mailbox_group'] = $this->mailbox_group;
		return true;
	}
	
	/* Extends parent to enforce that users may only edit their own flags */
	protected function _run_before_update($id, &$values){ 
		if(!parent::_run_before_update($id, $values)) return false;
		
		//make sure that the mailbox_group we're currently in matches the mailbox_group for the flag
		//this should ensure that only people with access to this mailbox are able to edit flags.
		if(!$this->exists(array($this->primary_key => $id, 'mailbox_group' => $this->mailbox_group))){
			return $this->error->warning( 'Flag#'.$id.' does not belong to the '.$this->error->describe($this->mailbox_group).' mailbox group' );
		}
		
		return true; 
	}

	
	/* Extends parent to enforce that flag color is required */
	function _values_are_valid_for_create($values){
		if(!parent::_values_are_valid_for_create($values)) return false;
		if(!array_key_exists('color', $values) || empty($values['color'])) return $this->error->warning('Flag color is required');			
		return true;
	}
	
	/* Extends parent to enforce that flag color matches the known/allowed colors. */
	function _values_are_valid_for_create_and_update($values){ 
		if(!parent::_values_are_valid_for_create_and_update($values)) return false;
		if(array_key_exists('color', $this->colors) && (empty($values['color']) || !in_array($values['color'], $this->colors)))
			return $this->error->should_be_a_known_color($values['color']);
		if(array_key_exists('mailbox_group', $values) && $this->mailbox_group != $values['mailbox_group']){
			return $this->error->warning( 'The mailbox group for flag#'.$id.' must match the current mailbox_group ('.
										  $this->error->describe($this->mailbox_group).'), but you gave me '.$this->error->describe($values['mailbox_group']));
		}	
		
		return true;
	}
	
	protected function _clear_record($record_id){
		if(!$this->is->nonzero_unsigned_integer($record_id)) return $this->error->should_be_a_nonzero_unsigned_integer($record_id);
		$values = array('color' => null, 'content' => null);
		
		//since this is the only occasion that we can set a color null, we'll circumvent the normal validation in update() 
		//just run the methods to auto-generate values as necessary
		$this->_run_before_update($record_id, $values);
		$this->_run_before_create_and_update($values, $record_id);
		return $this->_update($record_id, $values);
	}
	
}